from typing import List

from Slot import *
from Vincoli import *
from templates import *
from RulePenalty import * 
from auxiliary import AuxiliaryStruct
from log import logger
from dbAPI import dbAPI
from patterns import *


class SlotReqHandler(metaclass=SingletonMeta):
    '''Carica il design di un Insegnamento dal file xml.\n
    Params:
        aux (AuxiliaryStruct): classe gestore delle risorse ausiliarie per il modelllo
    '''
    def __init__(self):
        self.listSlotReq:List[SlotReq] = list()
        self.listVincoli:List[Vincolo] = list()
        self.listOperatori:List[Operatore] = list()
        
        self.endLoaded = False
        
        self.AUX = AuxiliaryStruct()
        self.dbAPI = dbAPI()
        self.log:logger = logger()
        
    def createId(self, idInsegnamento:int, baseId:str) -> str:
        return self.AUX.list_metaInsegnamenti[idInsegnamento].ID_INC + "_" + baseId
        
    def startNewInsegnamento(self):
        '''Cancella i dati dell'Insegnamento precedente'''        
        self.listOperatori.clear()
        self.listSlotReq.clear()
        self.listVincoli.clear()
            
    def loadInsegnamento(self, xmlTemplateOrario, idInsegnamento:int):
        '''Caricamento degli Slot e dei Vincoli dal file xml contenente l'Insegnamento. Compliant con schemaUseCase.xsd.\n
        Args:
            xmlTemplateOrario (XML Element): xml element di TemplateOrario
            idInsegnamento (int): id dell'Insegnamento all'interno del modello CPLEX (definito esternamente)
        '''
        if self.endLoaded:
            self.log.error_log("loadInsegnamento() non può più essere chiamato")
            exit(-1)
        self.startNewInsegnamento()
                
        # precedenza alla richiesta esplicita del docente
        numIscritti:int = 0
        for xmlSlot in xmlTemplateOrario:
            if xmlSlot.tag == "CapienzaAula":
                numIscritti = getNumStudentiFromCapienza(getCapienzaLocaleFromString(xmlSlot.text))
        if numIscritti == 0:
            numIscritti = int(xmlTemplateOrario[2].text)
        if numIscritti == 0:
            # lo prendo dal db
            lIns = self.dbAPI.get_Insegnamento(int(xmlTemplateOrario[1].text))
            if len(lIns) == 0:
                self.log.error_log("SlotReqHandler.loadInsegnamento(): Insegnamento non presente nel db, ID_INC: " + str(xmlTemplateOrario[1].text))
            numIscritti = int(lIns[0][2])
        if numIscritti == 0:
            self.log.warning_log("SlotReqHandler.loadInsegnamento(): non sono riuscito a capire il numero di studenti iscritti a " + str(xmlTemplateOrario[0].text))
        
        for xmlSlot in xmlTemplateOrario:
            if xmlSlot.tag == "SlotGenerico":
                slotR = self.xmlLoadSlotGenerico(xmlSlot, numIscritti, idInsegnamento)
                if slotR is not None:
                    self.addSlotR(slotR)
            elif xmlSlot.tag == "SlotScelto":
                slotR = self.xmlLoadSlotScelto(xmlSlot, numIscritti, idInsegnamento)
                if slotR is not None:
                    self.addSlotR(slotR)
            elif xmlSlot.tag == "Vincolo":
                # continue
                vincolo = self.xmlLoadVincolo(xmlSlot, idInsegnamento)
                if vincolo is not None:
                    self.listVincoli.append(vincolo)
            elif xmlSlot.tag == "Operatore":
                operatore = self.xmlLoadOperatore(xmlSlot, idInsegnamento, True)
                if operatore is not None:
                    self.listOperatori.append(operatore)

        # add Slot to pianoAllocazione                
        for slot in self.listSlotReq:
            self.AUX.add_slotToPianoAllocazione(slot) # lista con tutti gli slot da allocare
        for vincolo in self.listVincoli:
            self.AUX.add_vincoloToListVincoliInsegnamenti(vincolo)
        for operatore in self.listOperatori:
            self.AUX.add_operatoreToListOperatoriInsegnamenti(operatore)
       
    
    def loadInsegnamentoFromDb(self, pianoAllocazione:str, ID_INC:int, idInsegnamento:int):
        '''Carica un Insegnameno come lista di SlotScelti partendo dal db'''
        if self.endLoaded:
            self.log.error_log("SlotReqHandler.loadInsegnamentoFromDb(): non posso più essere chiamato")
            exit(-1)
        
        self.startNewInsegnamento()
        slotsDataDb = self.dbAPI.get_pianoAllocazioneID_INC_withDocenti(pianoAllocazione, ID_INC)
        
        for slot_index in range(len(slotsDataDb)):
            slot = slotsDataDb[slot_index]
            
            # se cambia lo Slot
            if slot_index == 0 or slotsDataDb[slot_index-1][1] != slot[1]:
                slotR = SlotReq(TipoSlot.SlotScelto, slot[1], int(slot[2]), idInsegnamento, int(slot[4]))
                if slotR.numIscritti == 0:
                    self.log.warning_log("SlotReqHandler.loadInsegnamentoFromDb(): caricato uno Slot senza info sul numero di iscritti: " + str(ID_INC))
                slotR.setIdSlotAllocato(slot[6], slot[7])
                slotR.tipoLezione = getTipoLezioneFromStr(slot[3])
                slotR.tipoLocale = getTipoLocaleFromString(slot[8])
                slotR.tipoErogazione = getTipoErogazioneFromString(slot[9])
            
            # aggiungo solo i Docenti allo slot già allocato
            slotR.addDocente(self.AUX.map_strDocenti_to_idDocenti[slot[10]])
            
            # se cambia lo Slot
            if slot_index+1 == len(slotsDataDb) or slotsDataDb[slot_index+1][1] != slot[1]:
                self.addSlotR(slotR)

        # add Slot to pianoAllocazione                
        for slot in self.listSlotReq:
            self.AUX.add_slotToPianoAllocazione(slot) # lista con tutti gli slot da allocare        

    
    def addSlotR(self, slotR: SlotReq):
        '''aggiunta di un singolo Slot (Generico, Scelto, Preferito) per l'Insegnamento corrente'''
        if slotR.tipoSlot in [TipoSlot.SlotScelto, TipoSlot.SlotScelto_giorno]:
            if slotR.daySlotAllocato == -1:
                self.log.error_log("ListSlotReq.addSlotR(): il tipo di slot prevede slotR.daySlotAllocato assegnato")
                return
        if slotR.tipoSlot in [TipoSlot.SlotScelto, TipoSlot.SlotScelto_fasciaOraria]:
            if slotR.hourSlotAllocato == -1:
                self.log.error_log("ListSlotReq.addSlotR(): il tipo di slot prevede slotR.hourSlotAllocato assegnato")
                return
        self.listSlotReq.append(slotR)   
      
      
    def xmlLoadVincolo(self, xmlVincolo, idInsegnamento:int) -> Vincolo:
        '''Carica un Vincolo'''
        if xmlVincolo.tag != "Vincolo":
            self.log.error_log("xml element non di tipo SlotGenerico")
            return None
        for cc in xmlVincolo:
            if cc.tag == "TipoVincolo":
                tipoVincolo:TipoVincolo = getTipoVincoloFromStr(cc.text)
            elif cc.tag == "slotId1":
                slotId_iStr:str = cc.text
            elif cc.tag == "slotId2":
                slotId_jStr:str = cc.text
            elif cc.tag == "Penalita":
                penalita:LV_Penalties = getLV_PenaltiesFromString(cc.text)
          
        # se non sono vincoli globali => espansione dello slotId  
        if xmlVincolo.get('globalSlot1Id') != "true":
            slotId_iStr = self.createId(idInsegnamento,slotId_iStr)
        else:
            self.log.warning_log("SlotReqHanlder.xmlLoadVincolo(): usato uno slotId globale: " + slotId_iStr)
        if xmlVincolo.get('globalSlot2Id') != "true":
            slotId_jStr = self.createId(idInsegnamento,slotId_jStr)
        else:
            self.log.warning_log("SlotReqHanlder.xmlLoadVincolo(): usato uno slotId globale: " + slotId_jStr)
                
        vincolo = Vincolo(self.createId(idInsegnamento,str(xmlVincolo.get('vincoloId'))), tipoVincolo, slotId_iStr, slotId_jStr, penalita)
        return vincolo
        
    def xmlLoadOperatore(self, xmlOpeatore, idInsegnamento:int, root:bool = False) -> Operatore:
        '''Carica un Operatore'''
        if xmlOpeatore.tag != "Operatore":
            self.log.error_log("xml element non di tipo SlotGenerico")
            return None
        
        tipoOperatore:str = str(xmlOpeatore.get('id'))

        if tipoOperatore == "NOT":
            penalita:LV_Penalties = LV_Penalties.LV_0
            # ho per forza un solo Operatore figlio
            child = None
            for cc in xmlOpeatore:
                if cc.tag == "Operatore" and child is None:
                    child = cc
                elif cc.tag == "Operatore" and child is not None:
                    self.log.error_log("SlotReqHandler.xmlLoadOperatore(): NOT: ha un singolo Operatore figlio")
                elif cc.tag == "Vincolo":
                    self.log.error_log("SlotReqHandler.xmlLoadOperatore(): NOT: non ha Vincoli figli diretti")
                elif cc.tag == "Penalita" and root:
                    penalita = getLV_PenaltiesFromString(cc.text)
            
            operatore:Operatore = Operatore(TipoOperatore.NOT, penalita, root)
            if child is None:
                self.log.error_log("SlotReqHandler.xmlLoadOperatore(): NOT: manca figlio")
                return None
            opChild = self.xmlLoadOperatore(child, idInsegnamento)
            if not operatore.set1Operatore(opChild):
                return None
            return operatore
        
        if tipoOperatore == "OR" or tipoOperatore == "AND":
            penalita:LV_Penalties = LV_Penalties.LV_0
            childVinc1 = None
            childVinc2 = None
            childOp1 = None
            childOp2 = None
            end:bool = False
            
            for cc in xmlOpeatore:
                if cc.tag == "Penalita" and root:
                    penalita = getLV_PenaltiesFromString(cc.text)
                elif cc.tag == "Operatore" and not end:
                    if childOp1 is None:
                        childOp1 = cc
                        if childVinc1 is not None:
                            end = True
                    else:
                        childOp2 = cc
                        end = True
                elif cc.tag == "Vincolo" and not end:
                    if childVinc1 is None:
                        childVinc1 = cc
                        if childOp1 is not None:
                            end = True
                    else:
                        childVinc2 = cc
                        end = True
            # a) 2 Vincoli come figli
            if childVinc1 is not None and childVinc2 is not None:
                vinc1:Vincolo = self.xmlLoadVincolo(childVinc1, idInsegnamento)
                vinc2:Vincolo = self.xmlLoadVincolo(childVinc2, idInsegnamento)
                if vinc1 is None or vinc2 is None:
                    self.log.error_log("SlotReqHandler.xmlLoadOperatore(): OR/AND: errore figli Vincoli")
                
                if tipoOperatore == "OR":
                    operatore:Operatore = Operatore(TipoOperatore.OR, penalita, root)
                    if not operatore.set2Vincoli(vinc1, vinc2):
                        self.log.error_log("SlotReqHandler.xmlLoadOperatore(): OR: errore creazione figli")
                    return operatore
                else:
                    operatore:Operatore = Operatore(TipoOperatore.AND, penalita, root)
                    if not operatore.set2Vincoli(vinc1, vinc2):
                        self.log.error_log("SlotReqHandler.xmlLoadOperatore(): AND: errore creazione figli")
                    return operatore
            # b) 1 Vincolo e 1 Operatore come figli
            if childVinc1 is not None and childOp1 is not None:
                vinc1:Vincolo = self.xmlLoadVincolo(childVinc1, idInsegnamento)
                op1:Operatore = self.xmlLoadOperatore(childOp1, idInsegnamento)                
                if vinc1 is None or op1 is None:
                    self.log.error_log("SlotReqHandler.xmlLoadOperatore(): OR/AND: errore figli Vincolo/Operatore")

                if tipoOperatore == "OR":
                    operatore:Operatore = Operatore(TipoOperatore.OR, penalita, root)
                    if not operatore.set1Vincolo1Operatore(vinc1, op1):
                        self.log.error_log("SlotReqHandler.xmlLoadOperatore(): OR: errore creazione figli")
                    return operatore
                else:
                    operatore:Operatore = Operatore(TipoOperatore.AND, penalita, root)
                    if not operatore.set1Vincolo1Operatore(vinc1, op1):
                        self.log.error_log("SlotReqHandler.xmlLoadOperatore(): AND: errore creazione figli")
                    return operatore   
            # c) 2 Operatori come figli
            if childVinc1 is not None and childOp1 is not None:
                op1:Operatore = self.xmlLoadOperatore(childOp1, idInsegnamento)                
                op2:Operatore = self.xmlLoadOperatore(childOp2, idInsegnamento)                
                if vinc1 is None or op1 is None:
                    self.log.error_log("SlotReqHandler.xmlLoadOperatore(): OR/AND: errore figli Operatori")

                if tipoOperatore == "OR":
                    operatore:Operatore = Operatore(TipoOperatore.OR, penalita, root)
                    if not operatore.set2Operatori(op1, op2):
                        self.log.error_log("SlotReqHandler.xmlLoadOperatore(): OR: errore creazione figli")
                    return operatore
                else:
                    operatore:Operatore = Operatore(TipoOperatore.AND, penalita, root)
                    if not operatore.set2Operatori(vinc1, op1):
                        self.log.error_log("SlotReqHandler.xmlLoadOperatore(): AND: errore creazione figli")
                    return operatore                          
                                      
    def xmlLoadSlotGenerico(self, xmlSlotGenerico, numIscritti:int, idInsegnamento:int) -> SlotReq:
        '''Crea uno SlotReq di tipo SlotGenerico.\n        
        Args:
            xmlSlotGenerico (xml element): radice dell'elemento SlotGenerico
            numScritti (int): numero di iscritti a livello di Insegnamento (per i singoli slot potrebbe variare)
            ID_INC (str): per generare uno slotId univoco
        Returns:
            SlotReq
        '''
        if xmlSlotGenerico.tag != "SlotGenerico":
            self.log.error_log("xml element non di tipo SlotGenerico")
            return None
        slotR = SlotReq(TipoSlot.SlotGenerico, 
                        self.createId(idInsegnamento,str(xmlSlotGenerico.get('slotId'))),
                        numIscritti, idInsegnamento)
        for cc in xmlSlotGenerico:
            if cc.tag == "Docente":
                if cc.text != "Docente": # altrimenti ho diversi conflitti inesistenti
                    slotR.addDocente(self.AUX.map_strDocenti_to_idDocenti[str(cc.text)])
            elif cc.tag == "NumSlotConsecutivi":
                slotR.numSlot = int(cc.text)
            elif cc.tag == "NumIscritti":
                slotR.numIscritti = int(cc.text)
            elif cc.tag == "Tipo":
                if cc.text == "EL":
                    slotR.tipoLezione = TipoLezione.EsercitazioneLaboratorio
                elif cc.text == "EA":
                    slotR.tipoLezione = TipoLezione.EsercitazioneAula
            elif cc.tag == "Locale":
                slotR.tipoLocale = getTipoLocaleFromString(cc.text)
            elif cc.tag == "TipoErogazione":
                if cc.text == "Remoto":
                    slotR.tipoErogazione = TipoErogazione.Remoto
            elif cc.tag == "PreseElettriche":
                slotR.preseElettriche = getPreseElettricheFromString(cc.text)
            elif cc.tag == "Squadra":
                slotR.squadra = getSquadraFromString(cc.text)
            elif cc.tag == "Nota":
                slotR.note = cc.text
        return slotR

    def xmlLoadSlotScelto(self, xmlSlotScelto, numIscritti:int, idInsegnamento:int) -> SlotReq:
        '''Crea uno SlotReq di tipo SlotScelto
        Args:
            xmlSlotGenerico (xml element): radice dell'elemento SlotScelto
            numScritti (int): numero di iscritti a livello di Insegnamento (per i singoli slot potrebbe variare)
            ID_INC (str): per generare no slotId univoco
        Returns:
            SlotReq
        '''
        if xmlSlotScelto.tag != "SlotScelto":
            self.log.error_log("xml element non di tipo SlotScelto")
            return None
        slotR = SlotReq(TipoSlot.SlotScelto,
                    self.createId(idInsegnamento,str(xmlSlotScelto.get('slotId'))),
                    numIscritti, idInsegnamento)
        strDay:str = ""
        strFasciaOraria:str = ""
        for cc in xmlSlotScelto:
            if cc.tag == "Docente":
                if cc.text != "Docente": # altrimenti ho diversi conflitti inesistenti
                    slotR.addDocente(self.AUX.map_strDocenti_to_idDocenti[str(cc.text)])
            elif cc.tag == "NumSlotConsecutivi":
                slotR.numSlot = int(cc.text)
            elif cc.tag == "Giorno":
                strDay = cc.text
            elif cc.tag == "FasciaOraria":
                strFasciaOraria = cc.text
            elif cc.tag == "NumIscritti":
                slotR.numIscritti = int(cc.text)
            elif cc.tag == "Tipo":
                if cc.text == "EL":
                    slotR.tipoLezione = TipoLezione.EsercitazioneLaboratorio
                elif cc.text == "EA":
                    slotR.tipoLezione = TipoLezione.EsercitazioneAula
            elif cc.tag == "Locale":
                slotR.tipoLocale = getTipoLocaleFromString(cc.text)        
            elif cc.tag == "TipoErogazione":
                if cc.text == "Remoto":
                    slotR.tipoErogazione = TipoErogazione.Remoto                            
            elif cc.tag == "PreseElettriche":
                slotR.preseElettriche = getPreseElettricheFromString(cc.text)  
            elif cc.tag == "Squadra":
                slotR.squadra = getSquadraFromString(cc.text)  
            elif cc.tag == "Nota":
                slotR.note = cc.text                                                
        slotR.setIdSlotAllocato(strDay, strFasciaOraria)
        return slotR
          
            